// Solution to Gunslinger
// Author: Thomas Beuman

// Time complexity: O(n log n)
// Memory: O(n)

// @EXPECTED_RESULTS@: CORRECT

/*
Solution outline:

Determine the convex hull of the Daltons.
If Luke or the hatch is strictly inside the hull, it is impossible.

Otherwise, add Luke and the hatch to the set and redetermine the convex hull.
If Luke or the hatch is now inside the hull, the straight path between them is safe.
Otherwise, the shortest safe path goes along the hull;
  try both the clockwise and the counterclockwise path, and take the minimum.
*/

#include <cstdio>
#include <cmath>
#include <algorithm>
using namespace std;

struct vertex { int x, y;};

vertex V[1002], Hull[1002], leftmost;

// Returns the Euclidean distance between A and B
double dis (vertex A, vertex B)
{ return sqrt((A.x-B.x)*(A.x-B.x) + (A.y-B.y)*(A.y-B.y));
}

// Returns the area (doubled) of triangle ABC; area is negative iff ABC is oriented clockwise
int area2 (vertex A, vertex B, vertex C)
{ return (A.x-B.x) * (A.y-C.y) - (A.x-C.x) * (A.y-B.y);
}

// Returns true if the angle A->leftmost->B is positive rather than negative
// In case of zero, returns true if A is closer to leftmost than B
bool operator < (vertex A, vertex B)
{ int t = area2(A, leftmost, B);
  return t < 0 || (t == 0 && (A.x < B.x || (A.x == B.x && A.y < B.y)));
}

// Determines the convex hull of V[] and puts it in Hull[]; returns the size of Hull[]
int convexhull (int n)
{ // Look for leftmost vertex
  int i, k = 0, m;
  for (i = 1; i < n; i++)
    if (V[i].x < V[k].x || (V[i].x == V[k].x && V[i].y < V[k].y))
      k = i;
  leftmost = V[k];
  sort(V, V + n); // Sort vertices by angle with leftmost vertex
  Hull[0] = V[0]; // Start with leftmost vertex
  m = 1; // Number of vertices in convex hull
  for (i = 1; i < n; i++)
  { // While the angle Hull[m-2]->Hull[m-1]->V[i] is more than 180 deg, remove Hull[m-1]
    while (m > 1 && area2(Hull[m-2], Hull[m-1], V[i]) <= 0)
      m--;
    Hull[m++] = V[i]; // Add vertex to hull
  }
  return m;
}

int main()
{ int cases, casenr, n, i, j, k, m;
  double d1, d2;
  vertex Luke, Hatch;
  scanf("%d\n", &cases);
  for (casenr = 1; casenr <= cases; casenr++)
  { // Read input
    scanf("%d %d\n%d %d\n%d\n", &Luke.x, &Luke.y, &Hatch.x, &Hatch.y, &n);
    for (i = 0; i < n; i++)
      scanf("%d %d\n", &V[i].x, &V[i].y);
    // Determine convexhull of Daltons
    m = convexhull(n);
    // Check if Luke or Hatch is (strictly) inside the hull
    for (i = 1; i <= m && area2(Hull[i-1], Hull[i%m], Luke) > 0; i++);
    for (j = 1; j <= m && area2(Hull[j-1], Hull[j%m], Hatch) > 0; j++);
    if (i > m || j > m)
    { printf("IMPOSSIBLE\n");
      continue;
    }
    // Add Luke and Hatch to the collection of vertices and redetermine convex hull
    V[n] = Luke;
    V[n+1] = Hatch;
    m = convexhull(n+2);
    // Look for edges that Luke and Hatch are on
    for (i = 1; i <= m && area2(Hull[i-1], Hull[i%m], Luke) > 0; i++);
    for (j = 1; j <= m && area2(Hull[j-1], Hull[j%m], Hatch) > 0; j++);
    // Luke is on edge (i-1)<->(i%m), Hatch on edge (j-1)<->(j%m)
    if (i == j || i > m || j > m) // They're on the same edge or one is inside the hull
      printf("%.3lf\n", dis(Luke, Hatch));
    else // Try both a counterclockwise and clockwise path along the hull
    { // Counterclockwise path
      d1 = dis(Luke, Hull[i%m]) + dis(Hull[j-1], Hatch);
      for (k = i%m; k != j-1; k = (k+1)%m)
        d1 += dis(Hull[k], Hull[(k+1)%m]);
      // Clockwise (counterclockwise when going from Hatch to Luke)
      d2 = dis(Hatch, Hull[j%m]) + dis(Hull[i-1], Luke);
      for (k = j%m; k != i-1; k = (k+1)%m)
        d2 += dis(Hull[k], Hull[(k+1)%m]);
      // Output minimum
      printf("%.3lf\n", min(d1, d2));
    }
  }
  return 0;
}

