/*  O(N log N) solution using Graham's scan.

    @EXPECTED_RESULTS@: CORRECT
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define FOR(i,a,b) for (i = a; i < b; ++i)
#define REP(i,n)   FOR(i,0,n)

typedef struct Point { double x, y; } Point;

/* Returns the Euclidean distance between two points. */
static double dist(const Point *p, const Point *q)
{
    return hypot(q->x - p->x, q->y - p->y);
}

/* Returns double the signed area of a triangle. */
static double area(const Point *p, const Point *q, const Point *r)
{
    return p->x*q->y - p->y*q->x +
           p->y*r->x - p->x*r->y +
           q->x*r->y - r->x*q->y;
}

/* Compare points by x-coordinate, then by y-coordinate: */
static int cmp1(const void *a, const void *b)
{
    const Point *p = a, *q = b;
    if (p->x != q->x) return p->x < q->x ? -1 : +1;
    if (p->y != q->y) return p->y < q->y ? -1 : +1;
    return 0;
}

static Point pivot;

static double angle(const Point *p)
{
    return atan2(p->y - pivot.y, p->x - pivot.x);
}

/* Compare points by angle relative to pivot: */
static int cmp2(const void *a, const void *b)
{
    double x = angle(a), y = angle(b);
    return (x < y) ? -1 : (x > y) ? + 1 : 0;
}

/* Remove duplicates from an ordered list of points. */
static int deduplicate(Point pts[], int npt)
{
    int i, j;
    for (i = j = 1; j < npt; ++i, ++j)
    {
        if (pts[i - 1].x == pts[j].x && pts[i - 1].y == pts[j].y)
        {
            fprintf(stderr, "Warning: duplicate point (%lf,%lf) in input!\n",
                            pts[j].x, pts[j].y);
            --i;
        }
        pts[i] = pts[j];
    }
    return i;
}

/* Compute convex hull in O(N log N) time using Graham's scan. */
static int convex_hull(Point pts[], int npt)
{
    int i, j;
    qsort(pts, npt, sizeof(*pts), cmp1);
    npt = deduplicate(pts, npt);  /* NOTE: this should not be necessary! */
    pivot = pts[0];
    qsort(pts + 1, npt - 1, sizeof *pts, cmp2);
    for (i = j = 1; j < npt; ++i, ++j)
    {
        while (i >= 2 && area(&pts[i - 2], &pts[i - 1], &pts[j]) < 0) --i;
        pts[i] = pts[j];
    }
    return i;
}

/* Return whether p lies strictly inside a convex polygon: */
static int point_in_polygon(const Point *p, const Point pts[], int npt)
{
    int i;
    REP(i, npt) if (area(p, &pts[i], &pts[(i + 1)%npt]) <= 0) return 0;
    return 1;
}

/* Returns whether any of the given points lie strictly left of the line p->q: */
static int left_of(const Point *p, const Point *q, const Point pts[], int npt)
{
    int i;
    REP(i, npt) if (area(p, q, &pts[i]) > 0) return 1;
    return 0;
}

/* Computes whether two convex polygons strictly overlap in O(np*nq) time: */
static int polygons_overlap(const Point ps[], int np, const Point qs[], int nq)
{
    /* Search for an axis of separation: */
    int i;
    REP(i, np) if (!left_of(&ps[i], &ps[(i + 1)%np], qs, nq)) return 0;
    REP(i, nq) if (!left_of(&qs[i], &qs[(i + 1)%nq], ps, np)) return 0;
    return 1;
}

/* Returns the circumference of a simple polygon: */
static double circumference(const Point pts[], int npt)
{
    double res = 0;
    int i;
    REP(i, npt) res += dist(&pts[i], &pts[(i + 1)%npt]);
    return res;
}

/* Returns the distance of moving from p to q and then back along one side of
   the convex hull; i.e. the circumference of the convex hull when it is
   partitioned along the line p->q (discarding the points on the right). */
static double dist_around(const Point *p, const Point *q, Point pts[], int npt)
{
    int nlpt = 2, i;
    double res;
    Point *lpts = malloc((npt + 2)*sizeof *lpts);
    lpts[0] = *p;
    lpts[1] = *q;
    REP(i, npt) if (area(p, q, &pts[i]) > 0) lpts[nlpt++] = pts[i];
    res = circumference(lpts, convex_hull(lpts, nlpt));
    free(lpts);
    return res;
}

static double min(double x, double y) { return x < y ? x : y; }

/* Returns the length of the shortest path from `p' to `q', such that no point
   of the path lies inside a triangle formed by any three of the `npt' points
   passed in `pts', or -1 if no such path exists. */
static double solve(const Point *p, const Point *q, Point pts[], int npt)
{
    Point pq[2] = { *p, *q };

    /* Compute convex hull of the Daltons, and see if escape is impossible
       because Luke or the Hatch lie strictly inside it: */
    npt = convex_hull(pts, npt);
    if ( point_in_polygon(p, pts, npt) ||
         point_in_polygon(q, pts, npt) ) return -1;

    /* Check if the direct path is possible; i.e. if it doesn't intersect
       the convex hull.  This is the shortest possible path. */
    if (!polygons_overlap(pq, 2, pts, npt)) return dist(p, q);

    /* Path is intersected by the convex hull; we need to move around the left
       or right side of it, but we can pick the shortest of the two: */
    return min( dist_around(p, q, pts, npt),
                dist_around(q, p, pts, npt) ) - dist(p, q);
}

int main()
{
    int cases = 0;
    scanf("%d", &cases);
    while (cases-- > 0)
    {
        Point p = { 0, 0 }, q = { 0, 0 }, *pts;
        int npt = 0, i;
        double answer;
        scanf("%lf %lf", &p.x, &p.y);
        scanf("%lf %lf", &q.x, &q.y);
        scanf("%d", &npt);
        pts = calloc(npt, sizeof *pts);
        REP(i, npt) scanf("%lf %lf", &pts[i].x, &pts[i].y);
        answer = solve(&p, &q, pts, npt);
        if (answer < 0)
            printf("IMPOSSIBLE\n");
        else
            printf("%.3lf\n", answer);
        free(pts);
    }
    return 0;
}
